(* ::Subsubsection:: *)
(*MaksutovCamera package*)

(* ::Text:: *)
(*This impements the Cassegrain Combination optical system*)

(* ::Text:: *)
(*Chapter*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/MaksutovCamera"}]*)

BeginPackage["GeometricOptics`Packages`MaksutovCamera`", {"GeometricOptics`"}]

Options[MaksutovCamera] = {OutputType->"Report", OutputLevel->"Full"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
*)
MaksutovCamera::usage="MaksutovCamera calculates parameters for a Maksutov Camera optical system.";

MaksutovCameraInputPanel::usage = "MaksutovCameraInputPanel gives the GUI panel to work with MaksutovCamera";

$MaksutovCameraInputVariables = {"f1", "diam", "\[Theta]"};

Begin["`Private`"]

MaksutovCamera[{f1_, diam_, theta_}, opts___]:= MaksutovCamera[f1, diam, theta, opts];

MaksutovCamera[f1_, diam_, theta_, OptionsPattern[]] := 
 Module[{r, indm, delta, DS1, s, Ni, DC1, sol, outputs, inPanel, outPanel},
 
		If[ArgumentsQ["MaksutovCamera", {$MaksutovCameraInputVariables, {f1, diam, theta}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID" -> 7, "PackageName" -> "MaksutovCamera", "Arguments" -> <|"f1" -> f1, "diam" -> diam, "\[Theta]" -> theta|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];

			r = diam/2;
			indm = {{1, 1.5187228, 1}, {1, 1.5222376, 1}, {1, 1.514322, 1}}; (* green, blue, red *)
			delta = 0.1 diam;
			DS1 = -(1/(32 f1^3));
			(* Meniscus *)
			Ni = indm[[1, 2]];
			s = delta (Ni^2 - 1)/(0.97 Ni^2);
			DC1 = (1/(8 Ni^2 r1^4) ((-1 + Ni) (-r1 + 1/(Ni (r1 - s)^3) ((r1 + (-1 + Ni) (s - delta))^2 (Ni (r1 - delta) + delta) (r1 + (-1 + Ni) (s + Ni s - Ni delta))))));
			r11 = Quiet@FirstCase[Replace[r1, Solve[DC1 + DS1 == 0, r1, Reals]], _?(# < 0 && Abs[#] > 100 &)];
			r12 = r11 - s;
			
			TotalAberrations[{r1, r1 - s, -2 f1}, {delta, \[Delta] f1}, Map[Append[#, -1] &, indm], {0, 0, 0}, r, 0, 0, -Infinity, x, theta, {\[Lambda]1, \[Lambda]2, \[Lambda]3}];
			
			sol = Quiet@FindRoot[{GOSphericalCoefficient == 0, GOComaCoefficient == 0}, {r1, r11}, {\[Delta], 1}, MaxIterations -> 1000, AccuracyGoal -> 2, PrecisionGoal -> 2];
			
			TotalAberrations[{ReplaceAll[r1, sol], ReplaceAll[r1, sol] - s, -2 f1}, {delta, ReplaceAll[\[Delta], sol] f1}, Map[Append[#, -1] &, indm], 
							{0, 0, 0}, r, 0, 0, -Infinity, x, theta, {\[Lambda]1, \[Lambda]2, \[Lambda]3}];
							
			outputs = {{"Radius of the mirror", -2 f1},
						{"Thickness of the corrector", "\[Beta]" diam},
						{"Distance between corrector and mirror", ReplaceAll[\[Delta], sol]*f1},
						{"Anterior radius of the corrector", ReplaceAll[r1, sol]},
						{"Posterior radius of the corrector", ReplaceAll[r1, sol] - s},
						{"Data relative to green light",""},
						{"Height of the image", GOimageHeight[[1, GOn]]},
						{"Focal of corrector", GOdistancegauss[[1, 2]]},
						{"Focal length", GOfocalLength[[1]]},
						{"Transv. spherical aberration", GOaberration[[1]]},
						{"Sagittal coma", GOcoma[[1]]},
						{"Astigmatism", GOastigmatism[[1]]},
						{"Data relative to blu light", ""},
						{"Height of the image", GOimageHeight[[2, GOn]]},
						{"Focal of corrector", GOdistancegauss[[2, 2]]},
						{"Focal length", GOfocalLength[[2]]},
						{"Transv. spherical aberration", GOaberration[[2]]},
						{"Sagittal coma", GOcoma[[2]]},
						{"Astigmatism", GOastigmatism[[2]]},
						{"Data relative to red light", ""},
						{"Height of the image", GOimageHeight[[3, GOn]]},
						{"Focal of corrector", GOdistancegauss[[3, 2]]},
						{"Focal length", GOfocalLength[[3]]},
						{"Transv. spherical aberration", GOaberration[[3]]},
						{"Sagittal coma", GOcoma[[3]]},
						{"Astigmatism", GOastigmatism[[3]]},
						{"Axial chromatism ", GOdistancegauss[[2, GOn]] - GOdistancegauss[[3, GOn]]}};

			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Focal length of the primary mirror" , "f1", f1},
							{"Diameter of primary mirror", "diam", diam},
							{"Field angle  in degree", "theta", theta}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];
				
			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "MaksutovCamera" &], "Description"], 
												"date" -> DateString[], 
												"function" -> "MaksutovCamera", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "MaksutovCamera" &], "Description"], 
													"date" -> DateString[], 
													"function" -> "MaksutovCamera", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"f1" , f1},
														 {"diam", diam},
														 {"\[Theta]", theta}},
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[TextCell[DeleteCases[outputs[[All, 2]], Alternatives["", Style[___]]], "Output"]]],

							
			(* Arguments are not correct *)
			MessageDialog["MaksutovCamera not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"];]];

MaksutovCameraInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "MaksutovCamera"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {f1, diam, theta} = ReplaceAll[$MaksutovCameraInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run "<>package<>" function", "Text"],
					Grid[{{"Focal length of the primary mirror" , "f1", Tooltip[InputField[Dynamic[f1], Alignment -> Center], "Insert the value of focal length of the primary mirror"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]}},
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {f1, diam, theta}], OutputType -> outputtype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{f1, diam, theta}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]